# Ultralytics YOLO 🚀, GPL-3.0 license
from copy import copy

import numpy as np
import torch
import torch.nn as nn

from ultralytics.utils.loss import BboxLoss
from ultralytics.utils.ops import xywh2xyxy
from ultralytics.utils.tal import TaskAlignedAssigner, dist2bbox, make_anchors
from copy import copy
import torch.nn.functional as F
from ultralytics.utils.ops import crop_mask, xyxy2xywh

# Criterion class for computing training losses
class Loss:

    def __init__(self):  # model must be de-paralleled

        device = 'cuda:0'  # get model device
        # h = model.args  # hyperparameters

        # m = model.model[-1]  # Detect() module
        self.bce = nn.BCEWithLogitsLoss(reduction='none')
        # self.hyp = h
        # self.stride = m.stride  # model strides 
        self.stride = torch.tensor([ 8., 16., 32.]).to(device)
        self.nc = 9  # number of classes
        self.no = 73
        self.reg_max = 16
        self.device = device
        self.hyp_box = 7.5
        self.hyp_cls = 0.5
        self.hyp_dfl = 1.5

        self.use_dfl = True
        # roll_out_thr = h.min_memory if h.min_memory > 1 else 64 if h.min_memory else 0  # 64 is default

        self.assigner = TaskAlignedAssigner(topk=10,
                                            num_classes=9,
                                            alpha=0.5,
                                            beta=6.0,
                                            eps=1e-9)
        self.bbox_loss = BboxLoss(self.reg_max).to(device)
        self.proj = torch.arange(16, dtype=torch.float, device=device)

    def preprocess(self, targets, batch_size, scale_tensor):
        if targets.shape[0] == 0:
            out = torch.zeros(batch_size, 0, 5, device=self.device)
        else:
            i = targets[:, 0]  # image index
            _, counts = i.unique(return_counts=True)
            out = torch.zeros(batch_size, counts.max(), 5, device=self.device)
            for j in range(batch_size):
                matches = i == j
                n = matches.sum()
                if n:
                    out[j, :n] = targets[matches, 1:]
            out[..., 1:5] = xywh2xyxy(out[..., 1:5].mul_(scale_tensor))
        return out

    def bbox_decode(self, anchor_points, pred_dist):
        if self.use_dfl:
            b, a, c = pred_dist.shape  # batch, anchors, channels
            pred_dist = pred_dist.view(b, a, 4, c // 4).softmax(3).matmul(self.proj.type(pred_dist.dtype).to(pred_dist.device))
            # pred_dist = pred_dist.view(b, a, 4, c // 4).softmax(3).matmul(self.proj.type(pred_dist.dtype))
            # pred_dist = pred_dist.view(b, a, c // 4, 4).transpose(2,3).softmax(3).matmul(self.proj.type(pred_dist.dtype))
            # pred_dist = (pred_dist.view(b, a, c // 4, 4).softmax(2) * self.proj.type(pred_dist.dtype).view(1, 1, -1, 1)).sum(2)
        return dist2bbox(pred_dist, anchor_points, xywh=False)

    def __call__(self, preds, batch):
        loss = torch.zeros(3, device=self.device)  # box, cls, dfl
        feats = preds[1] if isinstance(preds, tuple) else preds
        pred_distri, pred_scores = torch.cat([xi.view(feats[0].shape[0], self.no, -1) for xi in feats], 2).split(
            (self.reg_max * 4, self.nc), 1)

        pred_scores = pred_scores.permute(0, 2, 1).contiguous()
        pred_distri = pred_distri.permute(0, 2, 1).contiguous()

        dtype = pred_scores.dtype
        batch_size = pred_scores.shape[0]
        imgsz = torch.tensor(feats[0].shape[2:], device=self.device, dtype=dtype) * self.stride[0]  # image size (h,w)
        anchor_points, stride_tensor = make_anchors(feats, self.stride, 0.5)

        # targets
        targets = torch.cat((batch['batch_idx'].view(-1, 1), batch['cls'].view(-1, 1), batch['bboxes']), 1)
        targets = self.preprocess(targets.to(self.device), batch_size, scale_tensor=imgsz[[1, 0, 1, 0]])
        gt_labels, gt_bboxes = targets.split((1, 4), 2)  # cls, xyxy
        mask_gt = gt_bboxes.sum(2, keepdim=True).gt_(0)

        # pboxes
        pred_bboxes = self.bbox_decode(anchor_points, pred_distri)  # xyxy, (b, h*w, 4)

        _, target_bboxes, target_scores, fg_mask, _ = self.assigner(
            pred_scores.detach().sigmoid(), (pred_bboxes.detach() * stride_tensor).type(gt_bboxes.dtype),
            anchor_points * stride_tensor, gt_labels, gt_bboxes, mask_gt)

        target_bboxes /= stride_tensor
        target_scores_sum = max(target_scores.sum(), 1)

        # cls loss
        # loss[1] = self.varifocal_loss(pred_scores, target_scores, target_labels) / target_scores_sum  # VFL way
        loss[1] = self.bce(pred_scores, target_scores.to(dtype)).sum() / target_scores_sum  # BCE

        # bbox loss
        if fg_mask.sum():
            loss[0], loss[2] = self.bbox_loss(pred_distri, pred_bboxes, anchor_points, target_bboxes, target_scores,
                                              target_scores_sum, fg_mask)

        loss[0] *= self.hyp_box  # box gain
        loss[1] *= self.hyp_cls  # cls gain
        loss[2] *= self.hyp_dfl  # dfl gain

        return loss.sum() * batch_size, loss.detach()  # loss(box, cls, dfl)

# Criterion class for computing training losses
# class SegLoss(Loss):

#     def __init__(self, overlap=True):  # model must be de-paralleled
#         super().__init__()
#         self.nm = 32  # number of masks
#         self.overlap = overlap

#     def __call__(self, preds, batch):
#         # ================== [新增/修改 START] ==================
#         # 1. 获取当前模型输出所在的正确设备 (例如 cuda:0)
#         device = preds[0].device
        
#         # 2. 强制校准 Loss 类的所有设备属性
#         self.device = device
        
#         # 3. 关键：把 self.stride 也搬过来！(之前的报错就是因为它)
#         if hasattr(self, 'stride') and self.stride is not None:
#             self.stride = self.stride.to(device)
            
#         # 4. 把 self.proj (DFL用的) 也搬过来
#         if hasattr(self, 'proj') and self.proj is not None:
#             self.proj = self.proj.to(device)
#         # ================== [新增/修改 END] ==================

        
#         loss = torch.zeros(4, device=self.device)  # box, cls, dfl
#         # feats, pred_masks, proto = preds if len(preds) == 3 else preds[1]
#         # l_det = torch.cat([preds[0],preds[1]], dim=-1).permute(0, 3, 1, 2)
#         # m_det = torch.cat([preds[3],preds[4]], dim=-1).permute(0, 3, 1, 2)
#         # s_det = torch.cat([preds[6],preds[7]], dim=-1).permute(0, 3, 1, 2)
#         l_det = torch.cat([preds[1], preds[0]], dim=-1).permute(0, 3, 1, 2)
#         m_det = torch.cat([preds[4], preds[3]], dim=-1).permute(0, 3, 1, 2)
#         s_det = torch.cat([preds[7], preds[6]], dim=-1).permute(0, 3, 1, 2)
#         feats = [l_det, m_det, s_det]
#         l_mc = preds[2].permute(0,3,1,2).reshape(l_det.shape[0], 32, -1)
#         m_mc = preds[5].permute(0,3,1,2).reshape(l_det.shape[0], 32, -1)
#         s_mc = preds[8].permute(0,3,1,2).reshape(l_det.shape[0], 32, -1)
#         pred_masks = torch.cat([l_mc, m_mc, s_mc], dim=-1)
#         proto =preds[9]
#         batch_size, _, mask_h, mask_w = proto.shape  # batch size, number of masks, mask height, mask width
#         pred_distri, pred_scores = torch.cat([xi.view(feats[0].shape[0], self.no, -1) for xi in feats], 2).split(
#             (self.reg_max * 4, self.nc), 1)

#         # b, grids, ..
#         pred_scores = pred_scores.permute(0, 2, 1).contiguous()
#         pred_distri = pred_distri.permute(0, 2, 1).contiguous()
#         pred_masks = pred_masks.permute(0, 2, 1).contiguous()

#         dtype = pred_scores.dtype
#         imgsz = torch.tensor(feats[0].shape[2:], device=self.device, dtype=dtype) * self.stride[0]  # image size (h,w)
#         anchor_points, stride_tensor = make_anchors(feats, self.stride, 0.5)

#         # targets
#         try:
#             batch_idx = batch['labels']["targets_out"][:, 0].view(-1, 1)
#             targets = torch.cat((batch_idx, batch['labels']["targets_out"][:, 1].view(-1, 1), batch['labels']["targets_out"][:, 2:]), 1)
#             targets = self.preprocess(targets.to(self.device), batch_size, scale_tensor=imgsz[[1, 0, 1, 0]])
#             gt_labels, gt_bboxes = targets.split((1, 4), 2)  # cls, xyxy
#             mask_gt = gt_bboxes.sum(2, keepdim=True).gt_(0)
#         except RuntimeError as e:
#             raise TypeError('ERROR ❌ segment dataset incorrectly formatted or not a segment dataset.\n'
#                             "This error can occur when incorrectly training a 'segment' model on a 'detect' dataset, "
#                             "i.e. 'yolo train model=yolov8n-seg.pt data=coco128.yaml'.\nVerify your dataset is a "
#                             "correctly formatted 'segment' dataset using 'data=coco128-seg.yaml' "
#                             'as an example.\nSee https://docs.ultralytics.com/tasks/segmentation/ for help.') from e

#         # pboxes
#         pred_bboxes = self.bbox_decode(anchor_points, pred_distri)  # xyxy, (b, h*w, 4)

#         _, target_bboxes, target_scores, fg_mask, target_gt_idx = self.assigner(
#             pred_scores.detach().sigmoid(), (pred_bboxes.detach() * stride_tensor).type(gt_bboxes.dtype),
#             anchor_points * stride_tensor, gt_labels, gt_bboxes, mask_gt)

#         target_scores_sum = max(target_scores.sum(), 1)

#         # cls loss
#         # loss[1] = self.varifocal_loss(pred_scores, target_scores, target_labels) / target_scores_sum  # VFL way
#         loss[2] = self.bce(pred_scores, target_scores.to(dtype)).sum() / target_scores_sum  # BCE

#         if fg_mask.sum():
#             # bbox loss
#             loss[0], loss[3] = self.bbox_loss(pred_distri, pred_bboxes, anchor_points, target_bboxes / stride_tensor,
#                                               target_scores, target_scores_sum, fg_mask)
#             # masks loss
#             masks = batch['labels']["masks_out"].to(self.device).float()
#             if tuple(masks.shape[-2:]) != (mask_h, mask_w):  # downsample
#                 masks = F.interpolate(masks[None], (mask_h, mask_w), mode='nearest')[0]

#             for i in range(batch_size):
#                 if fg_mask[i].sum():
#                     mask_idx = target_gt_idx[i][fg_mask[i]]
#                     if self.overlap:
#                         gt_mask = torch.where(masks[[i]] == (mask_idx + 1).view(-1, 1, 1), 1.0, 0.0)
#                     else:
#                         gt_mask = masks[batch_idx.view(-1) == i][mask_idx]
#                     xyxyn = target_bboxes[i][fg_mask[i]] / imgsz[[1, 0, 1, 0]]
#                     marea = xyxy2xywh(xyxyn)[:, 2:].prod(1)
#                     mxyxy = xyxyn * torch.tensor([mask_w, mask_h, mask_w, mask_h], device=self.device)
#                     loss[1] += self.single_mask_loss(gt_mask, pred_masks[i][fg_mask[i]], proto[i], mxyxy,
#                                                      marea)  # seg loss
#         # WARNING: Uncomment lines below in case of Multi-GPU DDP unused gradient errors
#         #         else:
#         #             loss[1] += proto.sum() * 0 + pred_masks.sum() * 0
#         # else:
#         #     loss[1] += proto.sum() * 0 + pred_masks.sum() * 0

#         loss[0] *= self.hyp_box  # box gain
#         loss[1] *= self.hyp_box / batch_size  # seg gain
#         # loss[1] *= self.hyp.seg  # seg gain (新的独立权重)
#         loss[2] *= self.hyp_cls  # cls gain
#         loss[3] *= self.hyp_dfl  # dfl gain

#         return loss.sum() * batch_size, loss.detach()  # loss(box, cls, dfl)

#     def single_mask_loss(self, gt_mask, pred, proto, xyxy, area):
#         # Mask loss for one image
#         pred_mask = (pred @ proto.view(self.nm, -1)).view(-1, *proto.shape[1:])  # (n, 32) @ (32,80,80) -> (n,80,80)
#         loss = F.binary_cross_entropy_with_logits(pred_mask, gt_mask, reduction='none')
#         return (crop_mask(loss, xyxy).mean(dim=(1, 2)) / area).mean()
class SegLoss(Loss):

    def __init__(self, overlap=True): 
        super().__init__()
        self.nm = 32  # number of masks
        self.overlap = overlap
        # 定义 hyperparameters (如果没有 hyp 属性，提供默认值)
        self.hyp_box = 7.5
        self.hyp_cls = 0.5
        self.hyp_dfl = 1.5
        self.hyp_seg = 2.5  # [修复] 给分割损失一个合理的权重，不要用 box gain

    def __call__(self, preds, batch):
        # 1. 设备对齐 (QAT 模型必备)
        device = preds[0].device
        self.device = device
        if hasattr(self, 'stride') and self.stride is not None:
            self.stride = self.stride.to(device)
        if hasattr(self, 'proj') and self.proj is not None:
            self.proj = self.proj.to(device)

        # 2. 初始化 Loss
        loss = torch.zeros(4, device=self.device)  # box, cls, dfl, seg
        
        # 3. 解析模型输出 (针对你的 QAT 模型特定输出)
        # 假设顺序是 [box, cls, mask_coef] * 3 layers + [proto]
        # 注意：这里假设 l_det 拼接顺序是 [cls, box] (80+4)，请务必确认你的模型输出顺序！
        l_det = torch.cat([preds[1], preds[0]], dim=-1).permute(0, 3, 1, 2)
        m_det = torch.cat([preds[4], preds[3]], dim=-1).permute(0, 3, 1, 2)
        s_det = torch.cat([preds[7], preds[6]], dim=-1).permute(0, 3, 1, 2)
        feats = [l_det, m_det, s_det]
        
        # Mask Coefficients (Mask 系数)
        l_mc = preds[2].permute(0, 3, 1, 2).reshape(l_det.shape[0], 32, -1)
        m_mc = preds[5].permute(0, 3, 1, 2).reshape(l_det.shape[0], 32, -1)
        s_mc = preds[8].permute(0, 3, 1, 2).reshape(l_det.shape[0], 32, -1)
        pred_masks = torch.cat([l_mc, m_mc, s_mc], dim=-1) # (B, 32, n_anchors)
        
        # Proto Mask
        proto = preds[9] # (B, 32, 160, 160)
        batch_size, _, mask_h, mask_w = proto.shape 
        
        # 4. 生成 Anchor 和 预测分布
        # split((reg_max * 4, nc), 1) -> 前面是box分布，后面是cls分数
        pred_distri, pred_scores = torch.cat([xi.view(feats[0].shape[0], self.no, -1) for xi in feats], 2).split(
            (self.reg_max * 4, self.nc), 1)

        pred_scores = pred_scores.permute(0, 2, 1).contiguous()
        pred_distri = pred_distri.permute(0, 2, 1).contiguous()
        pred_masks = pred_masks.permute(0, 2, 1).contiguous() # (B, n_anchors, 32)

        dtype = pred_scores.dtype
        imgsz = torch.tensor(feats[0].shape[2:], device=self.device, dtype=dtype) * self.stride[0]
        anchor_points, stride_tensor = make_anchors(feats, self.stride, 0.5)

        # 5. [关键修复] 解析 Targets (适配扁平化 Batch)
        # 不再去 batch['labels'] 里找了，直接找 batch['batch_idx'] 等
        try:
            # 如果经过了 criterion 适配器，batch 里可能有嵌套，优先读取扁平 key
            if 'batch_idx' in batch:
                batch_idx = batch['batch_idx'].view(-1, 1)
                t_cls = batch['cls'].view(-1, 1)
                t_bboxes = batch['bboxes']
                masks_gt_all = batch['masks'] # (Total_N, 160, 160)
            else:
                # 兼容旧代码结构 (万一)
                batch_idx = batch['labels']['targets_out'][:, 0].view(-1, 1)
                t_cls = batch['labels']['targets_out'][:, 1].view(-1, 1)
                t_bboxes = batch['labels']['targets_out'][:, 2:]
                masks_gt_all = batch['labels']['masks_out']

            targets = torch.cat((batch_idx, t_cls, t_bboxes), 1)
            targets = self.preprocess(targets.to(self.device), batch_size, scale_tensor=imgsz[[1, 0, 1, 0]])
            gt_labels, gt_bboxes = targets.split((1, 4), 2)  # cls, xyxy
            mask_gt = gt_bboxes.sum(2, keepdim=True).gt_(0)
        except Exception as e:
            raise TypeError('ERROR ❌ Target parsing failed. Check your collate_fn.') from e

        # 6. 解码预测框
        pred_bboxes = self.bbox_decode(anchor_points, pred_distri)

        # 7. 正负样本匹配 (Task Aligned Assigner)
        _, target_bboxes, target_scores, fg_mask, target_gt_idx = self.assigner(
            pred_scores.detach().sigmoid(), 
            (pred_bboxes.detach() * stride_tensor).type(gt_bboxes.dtype),
            anchor_points * stride_tensor, 
            gt_labels, 
            gt_bboxes, 
            mask_gt
        )

        target_scores_sum = max(target_scores.sum(), 1)

        # 8. 计算 Cls Loss
        loss[2] = self.bce(pred_scores, target_scores.to(dtype)).sum() / target_scores_sum

        # 9. 计算 Box & Seg Loss (仅针对前景)
        if fg_mask.sum():
            # Box Loss
            loss[0], loss[3] = self.bbox_loss(pred_distri, pred_bboxes, anchor_points, target_bboxes / stride_tensor,
                                              target_scores, target_scores_sum, fg_mask)
            
            # [核心修复] Seg Loss 计算逻辑
            # 处理 GT Masks: 确保 dimension 匹配 Proto (160x160)
            if tuple(masks_gt_all.shape[-2:]) != (mask_h, mask_w):
                masks_gt_all = F.interpolate(masks_gt_all[None].float(), (mask_h, mask_w), mode='nearest')[0]
            
            # 遍历 Batch 计算 Mask Loss
            for i in range(batch_size):
                if fg_mask[i].sum():
                    # a. 找到当前图片对应的所有 GT Mask
                    # batch_idx.view(-1) == i 生成一个布尔掩码，选出属于这张图的物体
                    mask_i = masks_gt_all[batch_idx.view(-1) == i]
                    
                    # b. 找到正样本对应的 GT Index
                    # target_gt_idx[i] 是 assigner 返回的，指向当前图片的第几个 GT
                    mask_idx = target_gt_idx[i][fg_mask[i]]
                    
                    # c. [修复] 直接索引！不要用 overlap=True 的语义分割逻辑！
                    # 我们现在的 mask_i 是 (N_obj, 160, 160) 的 Instance Masks 堆叠
                    # 直接选出匹配的 mask 即可
                    gt_mask = mask_i[mask_idx] 
                    
                    # d. 计算单个图片的 Mask Loss
                    xyxyn = target_bboxes[i][fg_mask[i]] / imgsz[[1, 0, 1, 0]]
                    marea = xyxy2xywh(xyxyn)[:, 2:].prod(1)
                    mxyxy = xyxyn * torch.tensor([mask_w, mask_h, mask_w, mask_h], device=self.device)
                    
                    loss[1] += self.single_mask_loss(
                        gt_mask, 
                        pred_masks[i][fg_mask[i]], 
                        proto[i], 
                        mxyxy,
                        marea
                    )

        # 10. 应用权重 Gain
        loss[0] *= self.hyp_box 
        loss[2] *= self.hyp_cls 
        loss[3] *= self.hyp_dfl 
        loss[1] *= self.hyp_seg / batch_size # [修复] 使用专门的 seg gain (2.5)

        return loss.sum() * batch_size, loss.detach() 

    def single_mask_loss(self, gt_mask, pred, proto, xyxy, area):
        # 矩阵乘法生成 Mask: (N, 32) @ (32, 160*160) -> (N, 160, 160)
        pred_mask = (pred @ proto.view(self.nm, -1)).view(-1, *proto.shape[1:]) 
        
        # [关键修复]：加上 .float()
        # 你的 gt_mask 可能是 uint8 类型，必须转为 float32 才能做 BCE Loss
        loss = F.binary_cross_entropy_with_logits(pred_mask, gt_mask.float(), reduction='none')
        
        # 只计算框内的 Loss，并按面积归一化
        return (crop_mask(loss, xyxy).mean(dim=(1, 2)) / area).mean()